// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 */

#include <linux/module.h>
#include <linux/kobject.h>

#include <linux/gpio.h>
#include <linux/platform_device.h>
#include <linux/of_device.h>
#include <linux/of_platform.h>

#define MODULE_NAME "pctl"
#define PCTL_DEVICE_GET_ATTR(_name) \
	(&dev_attr_##_name.attr)

#define GPIO_HIGH	1
#define GPIO_LOW	0

struct power_control_dev {
	struct gpio_desc *power_control_gpio;
};

struct power_control_dev *pctlp;

static const struct of_device_id pctl_of_ids[] = {
	{ .compatible = "amazon,power_control", },
	{ },
};
MODULE_DEVICE_TABLE(of, pctl_of_ids);

static unsigned int pc_refcount;

/* refcount and GPIO changes need to be atomic.
 * This mutex is to guarantee that
 */
DEFINE_MUTEX(pc_mutex);

/**
 * Acquire managed GPIO, setting it to HIGH
 */
void power_control_acquire(void)
{
	if (!pctlp || !pctlp->power_control_gpio)
		return;

	mutex_lock(&pc_mutex);
	pc_refcount++;

	if (gpiod_get_direction(pctlp->power_control_gpio) != GPIOF_DIR_OUT)
		gpiod_direction_output(pctlp->power_control_gpio, GPIO_HIGH);
	else
		gpiod_set_value(pctlp->power_control_gpio, GPIO_HIGH);
	mutex_unlock(&pc_mutex);
}
EXPORT_SYMBOL(power_control_acquire);

/**
 * Release managed GPIO. If usage count reached 0 the GPIO will be set LOW
 */
void power_control_release(void)
{
	if (!pctlp || !pctlp->power_control_gpio)
		return;

	mutex_lock(&pc_mutex);
	if (pc_refcount > 0)
		pc_refcount--;

	if (pc_refcount == 0) {
		if (gpiod_get_direction(pctlp->power_control_gpio) != GPIOF_DIR_OUT)
			gpiod_direction_output(pctlp->power_control_gpio, GPIO_LOW);
		else
			gpiod_set_value(pctlp->power_control_gpio, GPIO_LOW);
	}

	mutex_unlock(&pc_mutex);

}
EXPORT_SYMBOL(power_control_release);

static ssize_t
gpio_state_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	int ret;

	mutex_lock(&pc_mutex);
	ret = scnprintf(buf, PAGE_SIZE, "%u\n", pctlp->power_control_gpio ?
			gpiod_get_value(pctlp->power_control_gpio) : 0);
	mutex_unlock(&pc_mutex);
	return ret;
}

static ssize_t
gpio_state_store(struct device *dev, struct device_attribute *attr,
		 const char *buf, size_t count)
{
	int ret = 0;
	unsigned long state;

	if (!pctlp->power_control_gpio)
		return count; /* No GPIO. Ignore call */

	ret = kstrtoul(buf, 10, &state);

	if (ret)
		return ret;

	mutex_lock(&pc_mutex);
	gpiod_set_value(pctlp->power_control_gpio, state > 0 ?
			GPIO_HIGH : GPIO_LOW);
	mutex_unlock(&pc_mutex);
	return count;
}
static DEVICE_ATTR_RW(gpio_state);

static ssize_t
refcnt_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	int ret;

	mutex_lock(&pc_mutex);
	ret = scnprintf(buf, PAGE_SIZE, "%u\n", pc_refcount);
	mutex_unlock(&pc_mutex);
	return ret;
}

static ssize_t
refcnt_store(struct device *dev, struct device_attribute *attr,
	     const char *buf, size_t count)
{
	ssize_t status = strlen(buf);

	if (sysfs_streq(buf, "inc"))
		power_control_acquire();
	else if (sysfs_streq(buf, "dec"))
		power_control_release();

	return status;
}
static DEVICE_ATTR_RW(refcnt);

static struct attribute *pctl_attrs[] = {
	PCTL_DEVICE_GET_ATTR(refcnt),
	PCTL_DEVICE_GET_ATTR(gpio_state),
	NULL
};
ATTRIBUTE_GROUPS(pctl);

static int power_control_probe(struct platform_device *pdev)
{
	int err = 0, ret;
	struct device *dev = &pdev->dev;

	dev_dbg(dev, "%s\n", __func__);
	pctlp = devm_kzalloc(dev, sizeof(*pctlp), GFP_KERNEL);
	if (!pctlp)
		return -ENOMEM;

	platform_set_drvdata(pdev, pctlp);
	/**
	 * We obtain the GPIO without setting direction to output (GPIOD_ASIS)
	 * because the RTOS might have set this GPIO and we don't want to
	 * counter that by setting the line low, and we don't want to set it
	 * high unconditionally either.
	 */
	pctlp->power_control_gpio =
		devm_gpiod_get_index(dev, "pctl", 0, GPIOD_ASIS);

	if (IS_ERR(pctlp->power_control_gpio)) {
		dev_warn(dev, "Power control GPIO not supplied");
		pctlp->power_control_gpio = NULL;
	}

	ret = devm_device_add_groups(dev, pctl_groups);
	if (ret) {
		dev_err(dev, "power control sysfs creation failed\n");
		goto out;
	}

	return 0;
out:
	dev_err(dev, "err %d\n", ret);
	return err;
}

static int power_control_remove(struct platform_device *pdev)
{
	dev_dbg(&pdev->dev, "%s driver unloaded\n", pdev->name);
	return 0;
}

static struct platform_driver pctl_driver = {
	.probe		= power_control_probe,
	.remove		= power_control_remove,
	.driver = {
		.name	= MODULE_NAME,
		.of_match_table = pctl_of_ids,
		.suppress_bind_attrs = true,
	}
};
module_platform_driver(pctl_driver);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Power control driver");
MODULE_AUTHOR("Ken Hsieh <khsieh@amazon.com>");
MODULE_AUTHOR("Avi Shukron <ashukro@amazon.com>");
